package com.agilex.healthcare.veteranappointment.datalayer.appointment;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.veteranappointment.datalayer.AbstractDao;
import com.agilex.healthcare.veteranappointment.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.veteranappointment.datalayer.validPo;
import com.agilex.healthcare.veteranappointment.domain.*;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestStatus;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.Query;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;
import java.util.*;

@Repository
public class AppointmentRequestDao extends AbstractDao {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(AppointmentRequestDao.class);
	private static final String CC_BOOKED_STATUS = "'" + AppointmentRequestStatus.SCHEDULED_IN_COMMUNITY.getName()+"' ";


	public VARAppointmentRequests getActiveAppointmentRequests(PatientIdentifiers identifiers, DateFilter filter) {
		VARAppointmentRequests appointmentRequests = new VARAppointmentRequests();
		List<String> appointmentIds = new LinkedList<>();
		for( PatientIdentifier identifier : identifiers ) appointmentIds.add(identifier.getUniqueId());

		Query constructedQuery = this.constructWhereClause(appointmentIds, filter);
		List<AppointmentRequestPo> retrievedRequests = this.executeForAppointmentRequests(constructedQuery);

		for (AppointmentRequestPo retrievedRequest : retrievedRequests) {
			appointmentRequests.add(retrievedRequest.create());
		}

		return appointmentRequests;
	}
	public int getSubmittedRequestCountByApptType(String patientId, DateFilter filter, String typeOfCareId, String locationId) {

		StringBuilder whereClause = new StringBuilder();
		whereClause.append(" where status = 'Submitted' ");

		Map<String, Object> parameters = new HashMap<>();

		addPatientIdCriteria(whereClause);
		addActiveRecordCriteria(whereClause);
		addDateFilterCriteria(filter, whereClause);
		addTypeOfCareIdCriteria(whereClause);
		addLocationIdCriteria(whereClause);

		setPatientIdParameter(patientId, parameters);
		setActiveFlagParameter(parameters);
		setDateFilterParameter(filter, parameters);
		setTypeOfCareIdParameter(typeOfCareId, parameters);
		setLocationIdParameter(locationId, parameters);

		String query = "select count(*) from AppointmentRequestPo " + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return Math.toIntExact((Long)constructedQuery.getSingleResult());
	}

	public VARAppointmentRequests getAppointmentRequests(AppointmentRequestFilter appointmentRequestFilter) {
		Query constructedQuery = this.constructWhereClause(appointmentRequestFilter);
		List<AppointmentRequestPo> retrievedRequests = this.executeForAppointmentRequests(constructedQuery);

        VARAppointmentRequests appointmentRequests = new VARAppointmentRequests();
		for (AppointmentRequestPo retrievedRequest : retrievedRequests) {
			appointmentRequests.add(retrievedRequest.create());
		}

		return appointmentRequests;
	}

	public VARAppointmentRequest getActiveAppointmentRequest(PatientIdentifiers identifiers, String appointmentRequestId) {
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where patient.userId in (:patientIds) and id = :appointmentRequestId and activeFlag = true ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		query.setParameter("patientIds", getUniqueIds(identifiers));

		AppointmentRequestPo po = this.executeForAppointmentRequest(query);
        VARAppointmentRequest request = (po == null)? null : po.create();

		return request;
	}

	private List<String> getUniqueIds(PatientIdentifiers identifiers) {
		List<String> retval = new LinkedList<>();
		for(PatientIdentifier identifier : identifiers) retval.add(identifier.getUniqueId());
		return retval;
	}

	public VARAppointmentRequest getAppointmentRequest(PatientIdentifiers identifiers, String appointmentRequestId) {
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where patient.userId in (:patientIds) and id = :appointmentRequestId ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		query.setParameter("patientIds", getUniqueIds(identifiers));


		AppointmentRequestPo po = this.executeForAppointmentRequest(query);
        VARAppointmentRequest request = (po == null)? null : po.create();

		return request;
	}

	public Set<VARDetailCode> getDetailCodes() {
		String selectDetailCodeQuery = "from DetailCodePo order by code";
		Query query = getQuery(selectDetailCodeQuery);
		@SuppressWarnings("unchecked")
		List<DetailCodePo> retrievedDetailCodes = query.getResultList();
		Set<VARDetailCode> codes = new HashSet<VARDetailCode>();

		for (DetailCodePo retrievedDetailCode : retrievedDetailCodes) {
			codes.add(retrievedDetailCode.create());
		}

		return codes;
	}

	public VARDetailCode getDetailCodeByCode(String code) {
		String selectDetailCodeQuery = "from DetailCodePo where code = :code";
		Query query = getQuery(selectDetailCodeQuery);
		query.setParameter("code", code);

		DetailCodePo detailCode = executeForDetailCode(query);
		return detailCode.create();
	}

	public VARAppointmentRequestMessages getAppointmentRequestMessagesByAppointmentRequestId(PatientIdentifiers identifiers, String appointmentRequestId) {
        VARAppointmentRequestMessages messages = new VARAppointmentRequestMessages();

        VARAppointmentRequest parentRequest = null;

		try {
			parentRequest = this.getAppointmentRequest(identifiers, appointmentRequestId);
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() != Status.NOT_FOUND.getStatusCode()) {
				throw e;
			}
		}

		if (parentRequest != null) {
			String selectDetailCodeQuery = "from AppointmentRequestMessagePo where appointmentRequestId = :appointmentRequestId";
			Query query = getQuery(selectDetailCodeQuery);
			query.setParameter("appointmentRequestId", appointmentRequestId);
			List<AppointmentRequestMessagePo> retrievedMessages = executeForAppointmentRequestMessages(query);

			for (AppointmentRequestMessagePo retrievedMessage : retrievedMessages) {
				messages.add(retrievedMessage.create());
			}
		}

		return messages;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public VARAppointmentRequestMessage saveAppointmentRequestMessage(PatientIdentifiers identifiers, VARAppointmentRequestMessage appointmentRequestMessage, boolean isProvider) {
		// Second attempts will save two instances of the appointment request -
		// parent request and the second request.
        VARAppointmentRequest appointmentRequest = this.getAppointmentRequest(identifiers, appointmentRequestMessage.getAppointmentRequestId());
		if(isProvider){
			appointmentRequest.setHasVeteranNewMessage(true);
		}
		else{
			appointmentRequest.setHasProviderNewMessage(true);
		}

        UserHistoryEntityManager<AppointmentRequestMessagePo> arMessageEntityManager = new UserHistoryEntityManager<AppointmentRequestMessagePo>(entityManager, AppointmentRequestMessagePo.TABLE_NAME, appointmentRequestMessage);
        AppointmentRequestMessagePo savedAppointmentRequestMessage = arMessageEntityManager.save(new AppointmentRequestMessagePo(appointmentRequestMessage));

        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));

		return savedAppointmentRequestMessage.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public VARAppointmentRequest updateAppointmentRequestMessageFlag(PatientIdentifiers identifiers, String appointmentRequestId, boolean isProvider){
        VARAppointmentRequest appointmentRequest = this.getAppointmentRequest(identifiers, appointmentRequestId);

		if(appointmentRequest == null)
			return appointmentRequest;

		if(isProvider){  //If user is provider, mark as provider read all new messages
			appointmentRequest.setHasProviderNewMessage(false);
		} else {
			appointmentRequest.setHasVeteranNewMessage(false);
		}
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo savedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));

		return savedAppointmentRequest.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public VARAppointmentRequest updateProviderSeenAppointmentRequestFlag(PatientIdentifiers identifiers, String appointmentRequestId){
        VARAppointmentRequest appointmentRequest = this.getAppointmentRequest(identifiers, appointmentRequestId);
		AppointmentRequestPo updatedAppointmentRequest = null;

        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);

		if(appointmentRequest != null && appointmentRequestId.equals(appointmentRequest.getAppointmentRequestId())){
			appointmentRequest.setProviderSeenAppointmentRequest(true);
			updatedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));
		}

		VARAppointmentRequest varApptReq = null ;
		if( updatedAppointmentRequest != null ) {

			varApptReq = updatedAppointmentRequest.create( ) ;
		}

		return( varApptReq ) ;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public VARAppointmentRequest updateAppointmentRequest(VARAppointmentRequest appointmentRequest) {
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo mergedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));
		return mergedAppointmentRequest.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public VARAppointmentRequest saveAppointmentRequest(VARAppointmentRequest appointmentRequest) {
		// Second attempts will save two instances of the appointment request -
		// parent request and the second request.
        VARAppointmentRequest existingAppointmentRequest = this.getAppointmentRequestIfItExists(appointmentRequest.getAppointmentRequestId());
		if (existingAppointmentRequest != null) {
			appointmentRequest.setHasProviderNewMessage(existingAppointmentRequest.isHasProviderNewMessage());
			appointmentRequest.setHasVeteranNewMessage(existingAppointmentRequest.isHasVeteranNewMessage());
		}
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo savedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));
        if (savedAppointmentRequest != null) {
    		return savedAppointmentRequest.create();
        }
        return null;
	}

	public VARAppointmentRequest getAppointmentRequestIfItExists(String appointmentRequestId) {
		AppointmentRequestPo arpo = null;
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where id = :appointmentRequestId ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		try{
			arpo = (AppointmentRequestPo) query.getSingleResult();
		} catch(NoResultException e) {
			LOGGER.debug(e);
			arpo = null;
		} catch (NonUniqueResultException e) {
			LOGGER.debug(e);
			arpo = null;
		}
		return (arpo == null) ? null : arpo.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deleteAppointmentRequest(VARAppointmentRequest appointmentRequest) {
		validPo<AppointmentRequestPo> po = new validPo<AppointmentRequestPo>(entityManager, new AppointmentRequestPo(appointmentRequest));
		if (po.isOk()) {
			AppointmentRequestPo appointmentRequestToRemove = po.getPo();
			appointmentRequestToRemove.setDeletedDate(new Date());
			appointmentRequestToRemove.setActiveFlag(false);
			this.updateAppointmentRequest(appointmentRequestToRemove.create());
		}
	}

	private Query constructWhereClause(List<String> patientIds, DateFilter filter) {
		StringBuilder whereClause = new StringBuilder();

		Map<String, Object> parameters = new HashMap<String, Object>();

		whereClause.append(" where  status != " + CC_BOOKED_STATUS);

		addPatientIdsCriteria(whereClause);
		addActiveRecordCriteria(whereClause);
		addDateFilterCriteria(filter, whereClause);

		whereClause.append(" order by hasVeteranNewMessage DESC, lastUpdatedDate DESC");

		setPatientIdsParameter(patientIds, parameters);
		setActiveFlagParameter(parameters);
		setDateFilterParameter(filter, parameters);

		String query = "from AppointmentRequestPo " + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private void setPatientIdParameter(String patientId, Map<String, Object> parameters) {
		parameters.put("patientId", patientId);
	}

	private void setPatientIdsParameter(List<String> patientIds, Map<String, Object> parameters) {
		parameters.put("patientIds", patientIds);
	}

	private void addPatientIdCriteria(StringBuilder whereClause) {
		whereClause.append(" and patient.userId = :patientId ");
	}

	private void addPatientIdsCriteria(StringBuilder whereClause) {
		whereClause.append(" and patient.userId in (:patientIds) ");
	}

	private void setStatusParameter(Map<String, Object> parameters) {
		parameters.put("status", CC_BOOKED_STATUS);
	}
	private void addStatusCriteria(StringBuilder whereClause) {
		whereClause.append(" and status = :" + CC_BOOKED_STATUS);
	}

	private void addLocationIdCriteria(StringBuilder whereClause) {
		whereClause.append(" and facility.facilityCode = :locationId");
	}

	private void setLocationIdParameter(String locationId, Map<String, Object> parameters) {
		parameters.put("locationId", locationId);
	}

	private void setDateFilterParameter(DateFilter filter, Map<String, Object> parameters) {

		if (filter == null) {
			return;
		}

		if (NullChecker.isNotNullish(filter.getStartDate())) {
			parameters.put("startDate", filter.getStartDate());
		}
		if (NullChecker.isNotNullish(filter.getEndDate())) {
			parameters.put("endDate", filter.getEndDate());
		}
	}

	private void addDateFilterCriteria(DateFilter filter, StringBuilder whereClause) {

		if (filter == null) {
			return;
		}

		if (NullChecker.isNotNullish(filter.getStartDate())) {
			whereClause.append(" and lastUpdatedDate >= :startDate ");
		}
		if (NullChecker.isNotNullish(filter.getEndDate())) {
			whereClause.append(" and lastUpdatedDate <= :endDate ");
		}
	}

	private Query constructWhereClause(AppointmentRequestFilter filter) {

		StringBuilder whereClause = new StringBuilder();
		Map<String, Object> parameters = new HashMap<String, Object>();

		whereClause.append(" where  status != " + CC_BOOKED_STATUS);

		addActiveRecordCriteria(whereClause);
		addFacilityNameCriteria(filter, whereClause);
		addParentSiteCodeCriteria(filter, whereClause);
		addDateFilterCriteria(filter, whereClause);

		whereClause.append(" order by case when hasProviderNewMessage=1 and providerSeenAppointmentRequest=1 then  hasProviderNewMessage  end DESC, lastUpdatedDate ASC");

		setActiveFlagParameter(parameters);
		setFacilityNameParameter(filter, parameters);
		setParentSiteCodeParameter(filter, parameters);
		setDateFilterParameter(filter, parameters);

		String query = "from AppointmentRequestPo " + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private void setFacilityNameParameter(AppointmentRequestFilter filter, Map<String, Object> parameters) {
		if (filter != null && NullChecker.isNotNullish(filter.getFacilityName())) {
			parameters.put("facilityName", filter.getFacilityName().toLowerCase());
		}
	}

	private void setParentSiteCodeParameter(AppointmentRequestFilter filter, Map<String, Object> parameters) {
		if (filter != null && NullChecker.isNotNullish(filter.getParentSiteCode())) {
			parameters.put("parentSiteCode", filter.getParentSiteCode());
		}
	}

	private void setActiveFlagParameter(Map<String, Object> parameters) {
		parameters.put("activeFlag", true);
	}

	private void setTypeOfCareIdParameter(String typeOfCareId, Map<String, Object> parameters) {
		parameters.put("typeOfCareId", typeOfCareId);
	}

	private void addParentSiteCodeCriteria(AppointmentRequestFilter filter, StringBuilder whereClause) {
		if (filter != null && NullChecker.isNotNullish(filter.getParentSiteCode())) {
			whereClause.append(" and facility.parentSiteCode = :parentSiteCode ");
		}
	}

	private void addFacilityNameCriteria(AppointmentRequestFilter filter, StringBuilder whereClause) {
		if (filter != null && NullChecker.isNotNullish(filter.getFacilityName())) {
			whereClause.append(" and lower(facility.name) = :facilityName ");
		}
	}

	private void addActiveRecordCriteria(StringBuilder whereClause) {
		whereClause.append(" and activeFlag = :activeFlag ");
	}

	private void addTypeOfCareIdCriteria(StringBuilder whereClause) {
		whereClause.append(" and typeOfCareId = :typeOfCareId ");
	}

	private Query setParametersForQuery(String query, Map<String, Object> parameters) {
		//LOGGER.debug(String.format(":::: Query: %s", query));
		Query constructedQuery = this.entityManager.createQuery(query);
		for (String key : parameters.keySet()) {
			constructedQuery.setParameter(key, parameters.get(key));
		}
		return constructedQuery;
	}

	private List<AppointmentRequestPo> executeForAppointmentRequests(Query query) {
		@SuppressWarnings("unchecked")
		List<AppointmentRequestPo> appointmentRequestResults = query.getResultList();

		return appointmentRequestResults;
	}
	private List<AppointmentRequestMessagePo> executeForAppointmentRequestMessages(Query query) {
		@SuppressWarnings("unchecked")
		List<AppointmentRequestMessagePo> appointmentRequestMessageResults = query.getResultList();

		return appointmentRequestMessageResults;
	}

	private AppointmentRequestPo executeForAppointmentRequest(Query query) {
		try {
			return (AppointmentRequestPo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private DetailCodePo executeForDetailCode(Query query) {
		try {
			return (DetailCodePo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private Query getQuery(String dbQuery) {
		Query query = this.entityManager.createQuery(dbQuery);
		return query;
	}

}
